//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Funktion der allgemeinen Statik
//+
//+ Beschreibung:	Funktionen der allgemeinen Statik zur Bestimmung von 
//+					Schnittgren von Durchlauftrgern mit 2 und 3 Feldern, 
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Funktionen fr Stahlbeton
//+
//+ Beschreibung:	Funktionen zur Berechnung von Biege- und Schubbewehrung
//+					fr Bauteile aus B25 und BSt 500 S
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Funktionen fr Stahlbau
//+
//+ Beschreibung:	Biegebemessung el/el fr IPE- und HEB-Profile (St 37)
//+					und Suchfunktion fr I-Profile bei Belastung durch ein
//+					Moment
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Funktionen fr Holzbau
//+
//+ Beschreibung:	Biegebemessung fr Nadelholz S10 und S13 und Berechnung von
//+					Knickzahlen omega
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Datum:			11.12.1999
//+
//+ Autor:			Serkan Karadag
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

#include "mcadincl.h"
#include "math.h"

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Deklarierung der Funktionen
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

// Durchlauftrger, 2 Felder, Gleichlast
LRESULT DLT2_AmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b,
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e);
LRESULT DLT2_AminFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e);

LRESULT DLT2_MmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e);

LRESULT DLT2_QmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e);

LRESULT DLT2_MminFunction(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d,
		LPCOMPLEXSCALAR e);

// Durchlauftrger, 3 Felder, Gleichlast
LRESULT DLT3_AmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e,
		LPCOMPLEXSCALAR f);

LRESULT DLT3_AminFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e,
		LPCOMPLEXSCALAR f);

LRESULT DLT3_MmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e,
		LPCOMPLEXSCALAR f);

LRESULT DLT3_QmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e,
		LPCOMPLEXSCALAR f);

LRESULT DLT3_MminFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d,
		LPCOMPLEXSCALAR e,LPCOMPLEXSCALAR f);

// Bemessung fr Biegung, Stahlbetonbau
LRESULT As_M_B25_S500Function(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR c,
		LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e);

// Schubbemessung, Stahlbetonbau
LRESULT As_Q_B25_S500Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR c,
		LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e);

// I-Profile suchen, Stahlbau
LRESULT  Su_Iy_IFunction(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b);

// Biegebemessung Elastisch-Elastisch, Stahlbau
LRESULT  BemMy_IPE_St37Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b);
LRESULT  BemMy_HEB_St37Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b);

// Knickzahlen fr Nadelholz
LRESULT  omegaFunction(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b);

// Biegebemessung
LRESULT  herf_M_S10Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b,
		 LPCOMPLEXSCALAR c);
LRESULT  herf_M_S13Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b,
		 LPCOMPLEXSCALAR c);

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+ Fehlermeldungstabelle 
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

// rot umrandete Fehlermeldung in Mathcad

char *myErrorMessageTable[4] =

{	"auerhalb des Anwendungsbereiches der kh-Tabellen",
	"Beanspruchung fr Querschnitt zu hoch",
	"Dieser Trger existiert nicht!",
	"unzulssiger Lambda - Wert"

};



//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Funktionsinfos der einzelnen Funktionen
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

FUNCTIONINFO DLT2_Amax =

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT2_Amax",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1 und l2, der stndigen Last g und "
	"der Verkehrslast p die maximalen Auflagerkrfte "
	"A bis C in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT2_AmaxFunction,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 4 Argumente
	4,

	// alle 4 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR} 

};

FUNCTIONINFO DLT2_Amin = 

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT2_Amin",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1 und l2, der stndigen Last g und "
	"der Verkehrslast p die minimalen Auflagerkrfte "
	"A bis C in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT2_AminFunction,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 4 Argumente
	4,

	// alle 4 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR} 

};

FUNCTIONINFO DLT2_Mmax = 

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT2_Mmax",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1 und l2, der stndigen Last g und "
	"der Verkehrslast p die Feldmomente M(1) und M(2) "
	"in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT2_MmaxFunction,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 4 Argumente
	4,

	// alle 4 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR} 

};

FUNCTIONINFO DLT2_Qmax =

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT2_Qmax",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1 und l2, der stndigen Last g und "
	"der Verkehrslast p die maximalen Querkrfte "
	"Q(A,re) bis Q(C,li) in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT2_QmaxFunction,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 4 Argumente
	4,

	// alle 4 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR} 

};

FUNCTIONINFO DLT2_Mmin = 

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT2_Mmin",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,g,p",

	// Beschreibung der Funktion: was liefert sie
	"Liefert fr einen Durchlaufttrger mit den "
	"Sttzweiten l1 und l2, der stndigen Last g und "
	"der Verkehrslast p das Sttzmoment M(B) ber "
	"der Mittelsttze.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT2_MminFunction,  

	// Rckgabewert ist ein komplexer Skalarwert. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_SCALAR,

	// die Funktion bentigt/verwendet 4 Argumente
	4,   

	// alle 4 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR} 

};

FUNCTIONINFO DLT3_Amax =

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT3_Amax",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,l3,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1, l2 und l3, der stndigen Last g und "
	"der Verkehrslast p die maximalen Auflagerkrfte "
	"A bis D in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT3_AmaxFunction,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 5 Argumente
	5,

	// alle 5 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, 
	  COMPLEX_SCALAR}

};

FUNCTIONINFO DLT3_Amin = 

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT3_Amin",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,l3,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1, l2 und l3, der stndigen Last g und "
	"der Verkehrslast p die minimalen Auflagerkrfte "
	"A bis D in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT3_AminFunction,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 5 Argumente
	5,

	// alle 5 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, 
	  COMPLEX_SCALAR}

};

FUNCTIONINFO DLT3_Mmax = 

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT3_Mmax",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,l3,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1, l2 und l3, der stndigen Last g und "
	"der Verkehrslast p die Feldmomente "
	"M(1) bis M(3) in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT3_MmaxFunction,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 5 Argumente
	5,

	// alle 5 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, 
	  COMPLEX_SCALAR}
};

FUNCTIONINFO DLT3_Qmax = 

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT3_Qmax",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,l3,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1, l2 und l3, der stndigen Last g und "
	"der Verkehrslast p die maximalen Querkrfte "
	"Q(A,re) bis Q(D,li) in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT3_QmaxFunction,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 5 Argumente
	5,

	// alle 5 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR, 
	  COMPLEX_SCALAR}
};

FUNCTIONINFO DLT3_Mmin = 

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"DLT3_Mmin",

	// Beschreibung der bentigten Funktionsparameter
	"l1,l2,l3,g,p",

	// Beschreibung der Funktion, was liefert sie
	"Liefert fr einen Durchlauftrger mit den "
	"Sttzweiten l1, l2 und l3, der stndigen Last g "
	"und der Verkehrslast p die Sttzmomente "
	"M(b) und M(c) in einem Vektor.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)DLT3_MminFunction,

	// Rckgabewert ist ein komplexer Skalarwert. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 5 Argumente
	5,

	// alle 5 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR,
	COMPLEX_SCALAR, COMPLEX_SCALAR} 

};

FUNCTIONINFO As_M_B25_S500 =

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"As_M_B25_S500",

	// Beschreibung der bentigten Funktionsparameter
	"M,b,h",

	// Beschreibung der Funktion, was liefert sie
	"Liefert die erforderliche Bewehrungsmenge in cm^2 fr ein Biegemoment "
	"in [kNm].",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)As_M_B25_S500Function,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_ARRAY,

	// die Funktion bentigt/verwendet 3 Argumente
	3,

	// alle 3 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR} 

};

FUNCTIONINFO As_Q_B25_S500 =

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"As_Q_B25_S500",

	// Beschreibung der bentigten Funktionsparameter
	"Q,min b,z",

	// Beschreibung der Funktion, was liefert sie
	"Liefert die erforderliche Bewehrungsmenge fr Balken in cm^2/m fr "
	"Querkraft in [kN]. Min b und innerer Hebelarm z in [cm].",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)As_Q_B25_S500Function,

	// Rckgabewert ist ein komplexes Feld. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_SCALAR,

	// die Funktion bentigt/verwendet 3 Argumente
	3,

	// alle 3 komplexe Skalare 
	{ COMPLEX_SCALAR, COMPLEX_SCALAR, COMPLEX_SCALAR} 

};

FUNCTIONINFO Su_Iy_I =

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"Su_Iy_I",

	// Beschreibung der bentigten Funktionsparameter
	"Iyerf",

	// Beschreibung der Funktion: was liefert sie
	"Liefert die erforderliche Nennhhe des Profils "
	"fr das gewnschte Flchenmoment 2.Grades Iyerf (cm^4) um die y-y Achse.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)Su_Iy_IFunction,

	// Rckgabewert ist ein komplexer Skalarwert. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_SCALAR,

	// die Funktion bentigt/verwendet 1 Argument
	1,

	// alle 1 komplexe Skalare
	COMPLEX_SCALAR

};

FUNCTIONINFO BemMy_IPE_St37 =

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"BemMy_IPE_St37",

	// Beschreibung der bentigten Funktionsparameter
	"Myd",

	// Beschreibung der Funktion: was liefert sie
	"Liefert die erforderlich Nennhhe des gewnschten Profils fr eine "
	"Belastung Myd (kNm) auf Design-Niveau, Stahlsorte St37. "
	"Bemessung nach dem Verfahren Elastisch-Elastisch (E-E).",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)BemMy_IPE_St37Function,

	// Rckgabewert ist ein komplexer Skalarwert. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_SCALAR,

	// die Funktion bentigt/verwendet 1 Argument
	1,

	// alle 1 komplexe Skalare 
	COMPLEX_SCALAR

};

FUNCTIONINFO BemMy_HEB_St37 =

{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"BemMy_HEB_St37",

	// Beschreibung der bentigten Funktionsparameter
	"Myd",

	// Beschreibung der Funktion: was liefert sie
	"Liefert die erforderlich Nennhhe des gewnschten Profils fr eine "
	"Belastung Myd (kNm) auf Design-Niveau, Stahlsorte St37. "
	"Bemessung nach dem Verfahren Elastisch-Elastisch (E-E).",


	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)BemMy_HEB_St37Function,

	// Rckgabewert ist ein komplexer Skalarwert. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_SCALAR,

	// die Funktion bentigt/verwendet 1 Argument
	1,

	// alle 1 komplexe Skalare 
	COMPLEX_SCALAR

};

FUNCTIONINFO omega =

	{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"omega",

	// Beschreibung der bentigten Funktionsparameter
	"Lambda",

	// Beschreibung der Funktion: was liefert sie
	"Knickzahl fr Vollholz.",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)omegaFunction,

	// Rckgabewert ist ein komplexer Skalarwert. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_SCALAR,

	// die Funktion bentigt/verwendet 1 Argument
	1,

	// alle 1 komplexe Skalare
	{ COMPLEX_SCALAR }

};

FUNCTIONINFO herf_M_S10 =

	{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"herf_M_S10",

	// Beschreibung der bentigten Funktionsparameter
	"M,b",

	// Beschreibung der Funktion: was liefert sie
	"Liefert die erforderlich Profilhhe h [cm] fr ein Rechteckquerschnitt "
	"der Breite b [cm] und der Gteklasse NH, S10. Belastung: Biegemoment "
	"M [kNm].",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)herf_M_S10Function,

	// Rckgabewert ist ein komplexer Skalarwert. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_SCALAR,

	// die Funktion bentigt/verwendet 2 Argumente
	2,

	// beide komplexe Skalare
	{ COMPLEX_SCALAR, COMPLEX_SCALAR }

};

FUNCTIONINFO herf_M_S13 =

	{

	// Name, ber die die Funktion in MathCad aufgerufen wird
	"herf_M_S13",

	// Beschreibung der bentigten Funktionsparameter
	"M,b",

	// Beschreibung der Funktion: was liefert sie
	"Liefert die erforderlich Profilhhe h [cm] fr ein Rechteckquerschnitt "
	"der Breite b [cm] und der Gteklasse NH, S13. Belastung: Biegemoment "
	"M [kNm].",

	// Zeiger auf den ausfhrbaren Quelltext, wenn der Benutzer den 
	// Namen der Funktion eingibt
	(LPCFUNCTION)herf_M_S13Function,

	// Rckgabewert ist ein komplexer Skalarwert. Verwendet wird spter 
	// nur der reale Teil!
	COMPLEX_SCALAR,

	// die Funktion bentigt/verwendet 2 Argumente
	2,

	// beide komplexe Skalare
	{ COMPLEX_SCALAR, COMPLEX_SCALAR }

};

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Die ausfhrbaren Quelltexte der einzelnen Funktionen
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

// Der ausfhrbare Quelltext der Funktion DLT2_Amax

LRESULT DLT2_AmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e)

{	double stutzm[3], querkraft[4], Aufl[3];
	double g, p, q, l1, l2;
	int i;
	
	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 3, 1, true, false );

	// bernahme der realen Teile der Argumente
	
	l1 = b->real;
	l2 = c->real;
	g  = d->real;
	p  = e->real;

	// Berechnung der Sttzmomente M(B) fr LFK 1-3

	q = g+p;
	
	stutzm[0]=-.125*q*(l2*l2-l1*l2+l1*l1);						// LFK 1

	stutzm[1]=(-.125*(pow(l1,3)*q+pow(l2,3)*g))/(l1+l2);		// LFK 2

	stutzm[2]=(-.125*(pow(l1,3)*g+pow(l2,3)*q))/(l1+l2);		// LFK 3

	// Berechnen der maximalen Querkrfte an freigeschnittenen Trgerabschnitten

	querkraft[0] = (q*l1/2)+(stutzm[1]/l1);		// Q(A,re) aus LFK 2

	querkraft[1] = (-q*l1/2)+(stutzm[0]/l1);	// Q(B,li) aus LFK 1

	querkraft[2] = (q*l2/2)-(stutzm[0]/l2);		// Q(B,re) aus LFK 1

	querkraft[3] = -((q*l2/2)+(stutzm[2]/l2));	// Q(C,li) aus LFK 3

	// Berechnung der maximalen Auflagerkrfte

	Aufl[0] = querkraft[0];
	Aufl[1] = -querkraft[1] + querkraft[2];
	Aufl[2] = -querkraft[3];

	// Rckgabe der Realteile des ErgebnissesFeldes

	for ( i=0; i<3; i++)

		a->hReal[0][i] = Aufl[i];

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion DLT2_Amin

LRESULT DLT2_AminFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e)

{	double stutzm[3], querkraft[4], Aufl[3];
	double g, p, q, l1, l2;
	int i;

	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 3, 1, true, false );

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	g  = d->real;
	p  = e->real;

	// Berechnung der Sttzmomente fr LFK 2-4

	q = g+p;
	
	stutzm[0] = (-.125*(pow(l1,3)*q+pow(l2,3)*g))/(l1+l2);		// LFK 2

	stutzm[1] = (-.125*(pow(l1,3)*g+pow(l2,3)*q))/(l1+l2);		// LFK 3

	stutzm[2] = -.125*g*(pow(l2,2)-l1*l2+pow(l1,2));			// LFK 4

	// Berechnung der minimalen Querkrfte an freigeschnittenen Trgerabschnitten

	querkraft[0] = (g*l1/2)+(stutzm[1]/l1);						// LFK 3

	querkraft[1] = (-g*l1/2)+(stutzm[2]/l1);					// LFK 4

	querkraft[2] = (g*l2/2)-(stutzm[2]/l2);						// LFK 4

	querkraft[3] = -((g*l2/2)+(stutzm[0]/l2));					// LFK 2

	// Berechnung der minimalen Auflagerkrfte

	Aufl[0] = querkraft[0];
	Aufl[1] = -querkraft[1] + querkraft[2];
	Aufl[2] = -querkraft[3];

	// Rckgabe der Realteile des ErgebnissesFeldes

	for ( i=0; i<3; i++)

		a->hReal[0][i] = Aufl[i];

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion DLT2_Mmax

LRESULT DLT2_MmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e)

{	double stutzm[2], querkraft[2], feldm[2];
	double g, p, q, l1, l2;
	int i;

	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 2, 1, true, false );

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	g  = d->real;
	p  = e->real;

	// Berechnung der Sttzmomente fr LFK 2 und 3

	q = g+p;

	stutzm[0]=(-.125*(pow(l1,3)*q+pow(l2,3)*g))/(l1+l2);		// LFK 2

	stutzm[1]=(-.125*(pow(l1,3)*g+pow(l2,3)*q))/(l1+l2);		// LFK 3

	// Berechnen der magebenden Querkrfte an freigeschnittenen Trgerabschnitten

	querkraft[0] = (q*l1/2)+(stutzm[0]/l1);			// Q(A,re) aus LFK 2

	querkraft[1] = -((q*l2/2)+(stutzm[1]/l2));		// Q(C,li) aus LFK 3

	// Berechnung der maximalen Feldmomente

	feldm[0] = pow(querkraft[0],2)/(2*q);			// LFK 2
	feldm[1] = pow(querkraft[1],2)/(2*q);			// LFK 3

	// Rckgabe der Realteile des ErgebnissesFeldes

	for ( i=0; i<2; i++)

		a->hReal[0][i] = feldm[i];

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funtkion DLT2_Qmax

LRESULT DLT2_QmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e)

{	double stutzm[3], querkraft[4];
	double g, p, q, l1, l2;
	int i;

	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 4, 1, true, false );

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	g  = d->real;
	p  = e->real;

	// Berechnung der Sttzmomente LFK 1-3

	q = g+p;

	stutzm[0]=-.125*q*(l2*l2-l1*l2+l1*l1);					// LFK 1

	stutzm[1]=(-.125*(pow(l1,3)*q+pow(l2,3)*g))/(l1+l2);	// LFK 2

	stutzm[2]=(-.125*(pow(l1,3)*g+pow(l2,3)*q))/(l1+l2);	// LFK 3

	// Berechnung der maximalen Querkrfte an freigeschnittenen Trgerabschnitten 

	querkraft[0] = (q*l1/2)+(stutzm[1]/l1);		// Q(A,re) aus LFK 2

	querkraft[1] = (-q*l1/2)+(stutzm[0]/l1);	// Q(B,li) aus LFK 1

	querkraft[2] = (q*l2/2)-(stutzm[0]/l2);		// Q(B,re) aus LFK 1

	querkraft[3] = -((q*l2/2)+(stutzm[2]/l2));	// Q(C,li) aus LFK 3

	// Rckgabe der Realteile des Ergebnisses

	for ( i=0; i<4; i++)

		a->hReal[0][i] = querkraft[i];

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funtkion DLT2_Mmin

LRESULT DLT2_MminFunction(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b, 
						  LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d,
						  LPCOMPLEXSCALAR e)

{	double stutzm, g, p, q, l1, l2;

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	g  = d->real;
	p  = e->real;


	// Berechnung des Sttzmomentes

	q = g+p;

	stutzm = -.125*q*(l2*l2-l1*l2+l1*l1);

	// Rckgabe des Ergebnisses

	a->real = stutzm;

	// kein Fehler aufgetreten

	return 0;

}


// Der ausfhrbare Quelltext der Funktion DLT3_Amax

LRESULT DLT3_AmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e,
		LPCOMPLEXSCALAR f)

{	double stutzm[6], querkraft[6], Aufl[4];
	double g, p, q, l1, l2, l3, nenner;
	int i;

	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 4, 1, true, false );

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	l3 = d->real;
	g  = e->real;
	p  = f->real;

	// Berechnung der Sttzmomente fr LFK 1-3

	q = g+p;

	nenner=(3*l2*l2+4*l1*l2+4*l3*l1+4*l3*l2);

	// LFK 1, M(b) und M(c)

	stutzm[0]=.25*(l2*pow(l3,3)*g-2*l2*pow(l1,3)*q-pow(l2,4)*q-2*l3*pow(l1,3)
		*q-2*l3*pow(l2,3)*q)/nenner;
	stutzm[1]=-.25*(2*l2*pow(l3,3)*g-l2*pow(l1,3)*q+pow(l2,4)*q+2*l1*pow(l2,3)
		*q+2*l1*pow(l3,3)*g)/nenner;

	// LFK 2, M(b) und M(c)

	stutzm[2]=-.25*(-l2*pow(l3,3)*q+2*l2*pow(l1,3)*g+pow(l2,4)*q+2*l3*pow(l1,3)
		*g+2*l3*pow(l2,3)*q)/nenner;
	stutzm[3]=.25*(-2*l2*pow(l3,3)*q+l2*pow(l1,3)*g-pow(l2,4)*q-2*l1*pow(l2,3)
		*q-2*l1*pow(l3,3)*q)/nenner;

	// LFK 3, M(b) und M(c)

	stutzm[4]=-.25*(2*l2*pow(l1,3)*q+pow(l2,4)*g-l2*pow(l3,3)*q+2*l3*pow(l2,3)
		*g+2*l3*pow(l1,3)*q)/nenner;
	stutzm[5]=-.25*(2*l1*pow(l2,3)*g+2*l1*pow(l3,3)*q-l2*pow(l1,3)*q+pow(l2,4)
		*g+2*l2*pow(l3,3)*q)/nenner;

	// Berechnung der maximalen Querkrfte an freigeschnittenen Trgerabschnitten

	querkraft[0] = (q*l1/2)+(stutzm[4]/l1);					// Q(A,re) aus LFK 3 

	querkraft[1] = -(q*l1/2)+(stutzm[0]/l1);				// Q(B,li) aus LFK 1 

	querkraft[2] = (q*l2/2)+((stutzm[1]-stutzm[0])/l2);		// Q(B,re) aus LFK 1 

	querkraft[3] = -(q*l2/2)+((stutzm[3]-stutzm[2])/l2);	// Q(C,li) aus LFK 2

	querkraft[4] = (q*l3/2)-(stutzm[3]/l3);					// Q(C,re) aus LFK 2

	querkraft[5] = -((q*l3/2)+(stutzm[5]/l3));				// Q(D,li) aus LFK 3


	// Berechnung der maximalen Auflagerkrfte

	Aufl[0] = querkraft[0];					// A
	Aufl[1] = -querkraft[1]+querkraft[2];	// B
	Aufl[2] = -querkraft[3]+querkraft[4];	// C
	Aufl[3] = -querkraft[5];				// D

	// Rckgabe der Realteile des ErgebnissesFeldes

	for ( i=0; i<4; i++)

		a->hReal[0][i] = Aufl[i];

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion DLT3_Amin

LRESULT DLT3_AminFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e,
		LPCOMPLEXSCALAR f)

{	double stutzm[6], querkraft[6],Aufl[4];
	double g, p, q, l1, l2, l3, nenner;
	int i;

	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 4, 1, true, false );

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	l3 = d->real;
	g  = e->real;
	p  = f->real;

	// Berechnung der Sttzmomente fr LFK 4-6

	q=g+p;

	nenner=(3*l2*l2+4*l1*l2+4*l3*l1+4*l3*l2);

	// LFK 4, M(b) und M(c)

	stutzm[0]=(-2*l2*pow(l1,3)*g-pow(l2,4)*q+l2*pow(l3,3)*g-2*l3*pow(l2,3)*q
		-2*l3*pow(l1,3)*g)/(4*nenner);
	stutzm[1]=-(2*l1*pow(l2,3)*q+2*l1*pow(l3,3)*g-l2*pow(l1,3)*g+pow(l2,4)*q
		+2*l2*pow(l3,3)*g)/(4*nenner);

	// LFK 5, M(b) und M(c)

	stutzm[2]=-(2*l2*pow(l1,3)*g+pow(l2,4)*g-l2*pow(l3,3)*q+2*l3*pow(l1,3)*g
		+2*l3*pow(l2,3)*g)/(4*nenner);
	stutzm[3]=-(2*l1*pow(l2,3)*g+2*l1*pow(l3,3)*q-l2*pow(l1,3)*g+pow(l2,4)*g
		+2*l2*pow(l3,3)*q)/(4*nenner);

	// LFK 6, M(b) und M(c)

	stutzm[4]=(l2*pow(l3,3)*g-2*l2*pow(l1,3)*q-pow(l2,4)*g-2*l3*pow(l1,3)*q
		-2*l3*pow(l2,3)*g)/(4*nenner);
	stutzm[5]=-(2*l2*pow(l3,3)*g-l2*pow(l1,3)*q+pow(l2,4)*g+2*l1*pow(l2,3)*g
		+2*l1*pow(l3,3)*g)/(4*nenner);

	// Berechnung der minimalen Querkrfte an freigeschnittenen Trgerabschnitten

	querkraft[0] = (g*l1/2)+(stutzm[0]/l1);					// Q(A,re) aus LFK 4

	querkraft[1] = -(g*l1/2)+(stutzm[2]/l1);				// Q(B,li) aus LFK 5

	querkraft[2] = (g*l2/2)+((stutzm[3]-stutzm[2])/l2);		// Q(B,re) aus LFK 5

	querkraft[3] = -(g*l2/2)+((stutzm[5]-stutzm[4])/l2);	// Q(C,li) aus LFK 6

	querkraft[4] = (g*l3/2)-(stutzm[5]/l3);					// Q(C,re) aus LFK 6

	querkraft[5] = -((g*l3/2)+(stutzm[1]/l3));				// Q(D,li) aus LFK 4

	// Minimale Auflagerkrfte

	Aufl[0] = querkraft[0];						//  min A
	Aufl[1] = -querkraft[1]+querkraft[2];		//  min B
	Aufl[2] = -querkraft[3]+querkraft[4];		//  min C
	Aufl[3] = -querkraft[5];					//  min D


	// Rckgabe der Realteile des ErgebnissesFeldes

	for ( i=0; i<4; i++)

		a->hReal[0][i] = Aufl[i];

	// kein Fehler aufgetreten

	return 0;

}

LRESULT DLT3_MmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e,
		LPCOMPLEXSCALAR f)

{	double stutzm[4], querkraft[3], feldm[3];
	double g, p, q, l1, l2, l3, nenner;
	int i;

	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 3, 1, true, false );

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	l3 = d->real;
	g  = e->real;
	p  = f->real;

	// Berechnung der Sttzmomente fr LFK 3 und 4

	q=g+p;

	nenner=(3*l2*l2+4*l1*l2+4*l3*l1+4*l3*l2);

	// LFK 3, M(b) und M(c)

	stutzm[0]=-.25*(2*l2*pow(l1,3)*q+pow(l2,4)*g-l2*pow(l3,3)*q+
		2*l3*pow(l2,3)*g+2*l3*pow(l1,3)*q)/nenner;
	stutzm[1]=-.25*(2*l1*pow(l2,3)*g+2*l1*pow(l3,3)*q-l2*pow(l1,3)*q+
		pow(l2,4)*g+2*l2*pow(l3,3)*q)/nenner;

	// LFK 4, M(b) und M(c)

	stutzm[2]=(-2*l2*pow(l1,3)*g-pow(l2,4)*q+l2*pow(l3,3)*g-
		2*l3*pow(l2,3)*q-2*l3*pow(l1,3)*g)/(4*nenner);
	stutzm[3]=-(2*l1*pow(l2,3)*q+2*l1*pow(l3,3)*g-l2*pow(l1,3)*g+
		pow(l2,4)*q+2*l2*pow(l3,3)*g)/(4*nenner);

	// Berechnung der maximalen Querkrfte an freigeschnittenen Trgerabschnitten

	querkraft[0] = (q*l1/2)+(stutzm[0]/l1);					// Q(A,re) aus LFK 3

	querkraft[1] = (q*l2/2)+((stutzm[3]-stutzm[2])/l2);		// Q(B,re) aus LFK 4

	querkraft[2] = -((q*l3/2)+(stutzm[1]/l3));				// Q(D,li) aus LFK 3

	// Maximale Feldmomente

	feldm[0] = pow(querkraft[0],2)/(2*q);					// max M(1)

	feldm[1] = (pow(querkraft[1],2)/(2*q))+stutzm[2];		// max M(2)

	feldm[2] = pow(querkraft[2],2)/(2*q);					// max M(3)

	// Rckgabe der Realteile des ErgebnissesFeldes

	for ( i=0; i<3; i++)

		a->hReal[0][i] = feldm[i];

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion DLT3_Qmax

LRESULT DLT3_QmaxFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
		LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e,
		LPCOMPLEXSCALAR f)

{	double stutzm[6], querkraft[6];
	double g, p, q, l1, l2, l3, nenner;
	int i;

	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 6, 1, true, false );

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	l3 = d->real;
	g  = e->real;
	p  = f->real;

	// Berechnung der Sttzmomente fr LFK 1-3

	q=g+p;

	nenner=(3*l2*l2+4*l1*l2+4*l3*l1+4*l3*l2);

	// LFK 1, M(b) und M(c)

	stutzm[0]=.25*(l2*pow(l3,3)*g-2*l2*pow(l1,3)*q-pow(l2,4)*q-2*l3*pow(l1,3)*q
		-2*l3*pow(l2,3)*q)/nenner;
	stutzm[1]=-.25*(2*l2*pow(l3,3)*g-l2*pow(l1,3)*q+pow(l2,4)*q+
		2*l1*pow(l2,3)*q+2*l1*pow(l3,3)*g)/nenner;

	// LFK 2, M(b) und M(c)

	stutzm[2]=-.25*(-l2*pow(l3,3)*q+2*l2*pow(l1,3)*g+pow(l2,4)*q+
		2*l3*pow(l1,3)*g+2*l3*pow(l2,3)*q)/nenner;
	stutzm[3]=.25*(-2*l2*pow(l3,3)*q+l2*pow(l1,3)*g-pow(l2,4)*q-
		2*l1*pow(l2,3)*q-2*l1*pow(l3,3)*q)/nenner;

	// LFK 3, M(b) und M(c)

	stutzm[4]=-.25*(2*l2*pow(l1,3)*q+pow(l2,4)*g-l2*pow(l3,3)*q+
		2*l3*pow(l2,3)*g+2*l3*pow(l1,3)*q)/nenner;
	stutzm[5]=-.25*(2*l1*pow(l2,3)*g+2*l1*pow(l3,3)*q-l2*pow(l1,3)*q+
		pow(l2,4)*g+2*l2*pow(l3,3)*q)/nenner;

	// Berechnen der maximalen Querkrfte an freigeschnittenen Trgerabschnitten

	querkraft[0] = (q*l1/2)+(stutzm[4]/l1);					// Q(A,re) aus LFK 3

	querkraft[1] = -(q*l1/2)+(stutzm[0]/l1);				// Q(B,li) aus LFK 1

	querkraft[2] = (q*l2/2)+((stutzm[1]-stutzm[0])/l2);		// Q(B,re) aus LFK 1

	querkraft[3] = -(q*l2/2)+((stutzm[3]-stutzm[2])/l2);	// Q(C,li) aus LFK 2

	querkraft[4] = (q*l3/2)-(stutzm[3]/l3);					// Q(C,re) aus LFK 2

	querkraft[5] = -((q*l3/2)+(stutzm[5]/l3));				// Q(D,li) aus LFK 3

	// Rckgabe der Realteile des ErgebnissesFeldes

	for ( i=0; i<6; i++)

		a->hReal[0][i] = querkraft[i];

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion DLT3_Mmin

LRESULT DLT3_MminFunction(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR b, 
						  LPCOMPLEXSCALAR c, LPCOMPLEXSCALAR d,
						  LPCOMPLEXSCALAR e, LPCOMPLEXSCALAR f)


{	double stutzm[2], q, l1, l2, l3, g, p, nenner;

	// bernahme der realen Teile der Argumente

	l1 = b->real;
	l2 = c->real;
	l3 = d->real;
	g  = e->real;
	p  = f->real;

	// Reservierung von Speicherplatz fr ErgebnisMatrix

	MathcadArrayAllocate (a, 2, 1, true, false);

	// Berechnung der Sttzmomente

	q=g+p;

	nenner=(3*l2*l2+4*l1*l2+4*l3*l1+4*l3*l2);

	// LFK 1, min M(B)
	stutzm[0]=-.25*(2*l2*pow(l1,3)*q+pow(l2,4)*q-l2*pow(l3,3)*g+
		2*l3*pow(l1,3)*q+2*l3*pow(l2,3)*q)/(nenner);

	// LFK 2, min M(C)
	stutzm[1]=-.25*(2*l1*pow(l2,3)*q+2*l1*pow(l3,3)*q-l2*pow(l1,3)*g
		+pow(l2,4)*q+2*l2*pow(l3,3)*q)/(nenner);

	// Rckgabe des Realteiles des Ergebnisses

	a->hReal[0][0]=stutzm[0];
	a->hReal[0][1]=stutzm[1];

	// kein Fehler aufgetreten

	return 0;

}

LRESULT As_M_B25_S500Function(LPCOMPLEXARRAY a, LPCOMPLEXSCALAR c, 
		LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e)

{	double As[2];

	double b;		// Breite
	double h;		// statische Hhe
	double M;		// Bemessungsmoment
	double pos;		// Hilfsvariable, Bemessungsmoment positiv oder negativ

	double d1;		// Abstand zum oberen Bauteilrand
	double d1h;		// Verhltnis d1 zu h

	double kh;		
	double ks;		

	double ks1;		
	double ks2;		
	double roh1;	
	double roh2;

	double khWert[10] = {7.9, 4.2, 3.0, 2.5, 2.22, 2.05, 1.93, 1.84, 1.78, 1.72};

	double ksWert[10] = {3.6, 3.7, 3.8, 3.9, 4.0, 4.1, 4.2, 4.3, 4.4, 4.5};

	double khstern[22]={ 1.72, 1.69, 1.67, 1.65, 1.63, 1.60, 1.57, 1.55, 1.52,
						1.5, 1.47, 1.45, 1.42, 1.39, 1.36, 1.33, 1.30, 1.27,
						1.24, 1.21, 1.18, 1.14};

	double ks1Wert[22] = {  0, .1, .2, .3, .4, .5, .6, .7, .8, .9, 1.0, 1.1, 1.2, 1.3,
						1.4, 1.5, 1.6, 1.7, 1.8, 1.9, 2.0, 2.1 };

	double ks2Wert[22] = {  4.5, 4.5, 4.5, 4.5, 4.4, 4.4, 4.4, 4.4, 4.4, 4.3, 4.3, 4.3,
						4.3, 4.3, 4.2, 4.2, 4.2, 4.2, 4.2, 4.1, 4.1, 4.1 };

	double d1hWert[10] = { 0, 0.07, 0.08, 0.1, 0.12, 0.14, 0.16, 0.18, 0.2, 0.22};

	double roh1Wert[10] = { 0, 1.0, 1.01, 1.03, 1.06, 1.08, 1.11, 1.15, 1.25, 1.37};

	double roh2Wert[5][10] ={	{0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0},
							{0, 1.0, 1.0, 1.0, 1.01, 1.01, 1.01, 1.01, 1.01, 1.02},
							{0, 1.0, 1.0, 1.01, 1.01, 1.02, 1.02, 1.03, 1.03, 1.04},
							{0, 1.0, 1.0, 1.01, 1.02, 1.03, 1.04, 1.04, 1.05, 1.06},
							{0, 1.0, 1.0, 1.02, 1.03, 1.04, 1.05, 1.06, 1.07, 1.09} };

	int i;
	int j;
	int Fall;

	// bernahme der Realteile der Argumente

	b = d -> real;
	h = e -> real;
	M = c -> real;
	pos = 1;

	// Aus negativ mach positiv

	if ( M < 0 )
		pos = -1;

	M = fabs(M);
	
	// Umrechnung b [cm] in [m]

	b = b / 100;

	// Berechnung der Bewehrungsmenge

	d1 = 4;					// Abstand obere Bewehrung zum oberen Rand in cm

	// kh-Wert

	kh = h/sqrt(M/b);

	if ( kh < khWert[9] )
		Fall = 1;			// mit Druckbewehrung

	else Fall = 2;			// ohne Druckbewehrung

	switch ( Fall )

	{	case 1:	// Auf Anwendbarkeit der Tabellen kontrollieren

				d1h = d1 / h;

				if ( kh < khstern[21] )
					return 1;

				if ( d1h > 0.22 )
					return 1;

				// ks1 und ks2 heraussuchen

				for (i=21; i>-1; i--)
					
					if ( kh >= khstern[i] )
					{	ks2 = ks2Wert[i];
						ks1 = ks1Wert[i];
					}

				if ( ks2 == 4.5 ) j=0;
				if ( ks2 == 4.4 ) j=1;
				if ( ks2 == 4.3 ) j=2;
				if ( ks2 == 4.2 ) j=3;
				if ( ks2 == 4.1 ) j=4;

				// roh1 und roh2 heraussuchen

				for (i=0; i<10; i++)

					if ( d1h > d1hWert[i] )
					{	roh2 = roh2Wert[j][i+1];
						roh1 = roh1Wert[i+1];
					}
						
				// Druckbewehrung

				As[0] = ks1*roh1*M/h;

				// Biegezugbewehrung

				As[1] = ks2*roh2*M/h;

				break;

		case 2:	// ks heraussuchen

				for (i=9; i>-1; i--)

					if ( kh >= khWert[i] )
						ks = ksWert[i];

				// Druckbewehrung

				As[0] = 0;

				// Biegezugbewehrung

				As[1] = ks*M/h;

				break;			
	}

	// Speicherplatz fr RckgabeMatrix reservieren

	MathcadArrayAllocate( a, 2, 1, true, false );

	// Ergebnis - Rckgabe

	a -> hReal[0][0] = As[0];
	a -> hReal[0][1] = As[1];

	if ( pos == -1 )
	{	a -> hReal[0][0] = As[1];
		a -> hReal[0][1] = As[0];
	}

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion As_Q_B25_S500

LRESULT As_Q_B25_S500Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR c, 
		LPCOMPLEXSCALAR d, LPCOMPLEXSCALAR e)

{	double as;		// Schubbewehrung

	double minb;	// geringste Breite im gezogenen Bereich
	double z;		// innerer Hebelarm
	double Qs;		// mgebende Querkraft

	double To;		// Grundwert der Schubspannung

	double limT[3] = { 0.75, 1.80, 3.00 };

	int SB;			// Schubbereich

	// bernahme der Argumente

	Qs   = c -> real;
	minb = d -> real;
	z    = e -> real;

	// Umrechnung von kN -> MN und cm -> m

	Qs   = Qs/1000;
	minb = minb/100;
	z    = z/100;

	// Grundwert der Schubspannung in MN/m^2

	To = Qs/(minb*z);

	// Einordnung in Schubbereich 1-3 bzw. Fehlermeldung

	if ( To > limT[2] )
		return 2;

	if ( To <= limT[2] )
		SB = 3;
	
	if ( To <= limT[1] )
		SB = 2;

	if ( To <= limT[0] )
		SB = 1;

	// Berechnung der Schubbewehrung

	switch ( SB )

	{	case 1:	as = 0.140 * (minb*100) * To;
				
				break;

		case 2:	as = 0.194 * (minb*100) * To * To;

				break;

		case 3: as = 0.350 * (minb*100) * To;

				break;
	}

	// Ergebnis - Rckgabe

	a -> real = as;

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion Su_Iy_I

LRESULT Su_Iy_IFunction(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b)

{	double I[20][2];
	double Iy;
	double h;
	int i;

	// bernahme der realen Teile der Argumente

	Iy = b->real;

	// Erstellen der ProfildatenMatrix

	// vorhandene Nennhhen					Flchenmomente 2.Grades 
	// I-Trger								um die y-y Achse

	I[0][0]  = 80;							I[0][1]  =    77.8;
	I[1][0]  =100;							I[1][1]  =   171;
	I[2][0]  =120;							I[2][1]  =   328;
	I[3][0]  =140;							I[3][1]  =   573;
	I[4][0]  =160;							I[4][1]  =   935;
	I[5][0]  =180;							I[5][1]  =  1450;
	I[6][0]  =200;							I[6][1]  =  2140;
	I[7][0]  =220;							I[7][1]  =  3060;
	I[8][0]  =240;							I[8][1]  =  4250;
	I[9][0]  =260;							I[9][1]  =  5740;
	I[10][0] =280;							I[10][1] =  7590;
	I[11][0] =300;							I[11][1] =  9800;
	I[12][0] =320;							I[12][1] = 12510;
	I[13][0] =340;							I[13][1] = 15700;
	I[14][0] =360;							I[14][1] = 19610;
	I[15][0] =400;							I[15][1] = 29210;
	I[16][0] =450;							I[16][1] = 45850;
	I[17][0] =500;							I[17][1] = 68740;
	I[18][0] =550;							I[18][1] = 99180;
	I[19][0] =600;							I[19][1] =139000;

	// Profil heraussuchen

	h = 0;
	
	for ( i=19; i>-1; i--)

	{	if ( I[i][1]>=Iy )
			h = I[i][0];
			
	}

	// Fehlermeldung, wenn gefordertes Profil nicht existiert!

	if ( h == 0 )
		return MAKELRESULT(1,3);

	// Rckgabe des Realteiles des Ergebnisses

	a->real = h;

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion BemMy_IPE_St37

LRESULT BemMy_IPE_St37Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b)

{	double IPE[18][2];
	double erfWy;
	double Myd;
	double zulSigma;
	double h;
	int i;

	zulSigma = 21.82;		// St37

	// bernahme der realen Teile der Argumente

	Myd = b->real;

	// Erstellen der ProfildatenMatrix

	// vorhandene Nennhhen						
	// IPE-Trger								Widerstandsmomente

	IPE[0][0]  = 80;							IPE[0][1]  =  20;
	IPE[1][0]  =100;							IPE[1][1]  =  34.2;
	IPE[2][0]  =120;							IPE[2][1]  =  53;
	IPE[3][0]  =140;							IPE[3][1]  =  77.3;
	IPE[4][0]  =160;							IPE[4][1]  = 109;
	IPE[5][0]  =180;							IPE[5][1]  = 146;
	IPE[6][0]  =200;							IPE[6][1]  = 194;
	IPE[7][0]  =220;							IPE[7][1]  = 252;
	IPE[8][0]  =240;							IPE[8][1]  = 324;
	IPE[9][0]  =270;							IPE[9][1]  = 429;
	IPE[10][0] =300;							IPE[10][1] = 557;
	IPE[11][0] =330;							IPE[11][1] = 713;
	IPE[12][0] =360;							IPE[12][1] = 904;
	IPE[13][0] =400;							IPE[13][1] =1160;
	IPE[14][0] =450;							IPE[14][1] =1500;
	IPE[15][0] =500;							IPE[15][1] =1930;
	IPE[16][0] =550;							IPE[16][1] =2440;
	IPE[17][0] =600;							IPE[17][1] =3070;

	// Profil heraussuchen

	h = 0;

	erfWy= (Myd*100) / zulSigma;
	
	for ( i=17; i>-1; i--)

	{	if ( IPE[i][1]>=erfWy )
			h = IPE[i][0];
			
	}

	// Fehlermeldung, wenn gefordertes Profil nicht existiert!

	if ( h == 0 )
		return MAKELRESULT(3,1);

	// Rckgabe des Realteiles des Ergebnisses

	a->real = h;

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion BemMy_HEB_St37

LRESULT BemMy_HEB_St37Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b)

{	double HEB[24][2];
	double erfWy;
	double Myd;
	double zulSigma;
	double h;
	int i;

	zulSigma = 21.82;		// St 37

	// bernahme der realen Teile der Argumente

	Myd = b->real;

	// Erstellen der ProfildatenMatrix

	// vorhandene Nennhhen						
	// HEB-Trger								Widerstandsmomente

	HEB[0][0]  = 100;							HEB[0][1]  =   89.9;
	HEB[1][0]  = 120;							HEB[1][1]  =  144;
	HEB[2][0]  = 140;							HEB[2][1]  =  216;
	HEB[3][0]  = 160;							HEB[3][1]  =  311;
	HEB[4][0]  = 180;							HEB[4][1]  =  426;
	HEB[5][0]  = 200;							HEB[5][1]  =  570;
	HEB[6][0]  = 220;							HEB[6][1]  =  736;
	HEB[7][0]  = 240;							HEB[7][1]  =  938;
	HEB[8][0]  = 260;							HEB[8][1]  = 1150;
	HEB[9][0]  = 280;							HEB[9][1]  = 1380;
	HEB[10][0] = 300;							HEB[10][1] = 1680;
	HEB[11][0] = 320;							HEB[11][1] = 1930;
	HEB[12][0] = 340;							HEB[12][1] = 2160;
	HEB[13][0] = 360;							HEB[13][1] = 2400;
	HEB[14][0] = 400;							HEB[14][1] = 2880;
	HEB[15][0] = 450;							HEB[15][1] = 3550;
	HEB[16][0] = 500;							HEB[16][1] = 4290;
	HEB[17][0] = 550;							HEB[17][1] = 4970;
	HEB[18][0] = 600;							HEB[18][1] = 5700;
	HEB[19][0] = 650;							HEB[19][1] = 6480;
	HEB[20][0] = 700;							HEB[20][1] = 7340;
	HEB[21][0] = 800;							HEB[21][1] = 8980;
	HEB[22][0] = 900;							HEB[22][1] =10980;
	HEB[23][0] =1000;							HEB[23][1] =12890;

	// Profil heraussuchen

	h = 0;

	erfWy= (Myd*100) / zulSigma;
	
	for ( i=23; i>-1; i--)

	{	if ( HEB[i][1]>=erfWy )
			h = HEB[i][0];
			
	}

	// Fehlermeldung, wenn gefordertes Profil nicht existiert!

	if ( h == 0 )
		return MAKELRESULT(3,1);

	// Rckgabe des Realteiles des Ergebnisses

	a->real = h;

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion omega

LRESULT omegaFunction(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b)

{	int i;
	double lambda;
	double omega;
	double Knickzahl[251] =
	
	
//				0		1		2		3		4		5		6		7		8		9

				{1.0,	1.0,	1.01,	1.01,	1.02,	1.02,	1.02,	1.03,	1.03,	1.04,	//   0
				1.04,	1.04,	1.05,	1.05,	1.06,	1.06,	1.06,	1.07,	1.07,	1.08,	//  10
				1.08,	1.09,	1.09,	1.10,	1.11,	1.11,	1.12,	1.13,	1.13,	1.14,	//  20
				1.15,	1.16,	1.17,	1.18,	1.19,	1.20,	1.21,	1.22,	1.24,	1.25,	//  30
				1.26,	1.27,	1.29,	1.30,	1.32,	1.33,	1.35,	1.36,	1.38,	1.40,	//  40
				1.42,	1.44,	1.46,	1.48,	1.50,	1.52,	1.54,	1.56,	1.58,	1.60,	//  50

				1.62,	1.64,	1.67,	1.69,	1.72,	1.74,	1.77,	1.80,	1.82,	1.85,	//  60
				1.88,	1.91,	1.94,	1.97,	2.00,	2.03,	2.06,	2.10,	2.13,	2.16,	//  70
				2.20,	2.23,	2.27,	2.31,	2.35,	2.38,	2.42,	2.46,	2.50,	2.54,	//  80
				2.58,	2.62,	2.66,	2.70,	2.74,	2.78,	2.82,	2.87,	2.91,	2.95,	//  90
				3.00,	3.06,	3.12,	3.18,	3.24,	3.31,	3.37,	3.44,	3.50,	3.57,	// 100

				3.63,	3.70,	3.76,	3.83,	3.90,	3.97,	4.04,	4.11,	4.18,	4.25,	// 110
				4.32,	4.39,	4.46,	4.54,	4.61,	4.68,	4.76,	4.84,	4.92,	4.99,	// 120
				5.07,	5.15,	5.23,	5.31,	5.39,	5.47,	5.55,	5.63,	5.71,	5.80,	// 130
				5.88,	5.96,	6.05,	6.13,	6.22,	6.31,	6.39,	6.48,	6.57,	6.66,	// 140
				6.75,	6.84,	6.93,	7.02,	7.11,	7.21,	7.30,	7.39,	7.49,	7.58,	// 150

				7.68,	7.78,	7.87,	7.97,	8.07,	8.17,	8.27,	8.37,	8.47,	8.57,	// 160
				8.67,	8.77,	8.88,	8.98,	9.08,	9.19,	9.29,	9.40,	9.51,	9.61,	// 170
				9.72,	9.83,	9.94,	10.05,	10.16,	10.27,	10.38,	10.49,	10.60,	10.72,	// 180
				10.83,	10.94,	11.06,	11.17,	11.29,	11.41,	11.52,	11.64,	11.76,	11.88,	// 190
				12.00,	12.12,	12.24,	12.36,	12.48,	12.61,	12.73,	12.85,	12.98,	13.10,	// 200

				13.23,	13.36,	13.48,	13.61,	13.74,	13.87,	14.00,	14.13,	14.26,	14.39,	// 210
				14.52,	14.65,	14.79,	14.92,	15.05,	15.19,	15.32,	15.46,	15.60,	15.73,	// 220
				15.87,	16.01,	16.15,	16.29,	16.43,	16.57,	16.71,	16.85,	16.99,	17.14,	// 230
				17.28,	17.42,	17.57,	17.71,	17.86,	18.01,	18.15,	18.30,	18.45,	18.60,	// 240
				18.75	};																		// 250

	// bernahme der realen Teile der Argumente

	lambda = b->real;

	// Ist Lambda innerhalb der Grenzen?

	if ( (lambda < 0) || (lambda > 250 ) )
		return MAKELRESULT(4,1);

	// Omega-Wert heraussuchen aus Knickzahl 

	for (i=0; i<251; i++)

		if (lambda <= i)

		{	omega = Knickzahl[i];
			break;
		}

	// Rckgabe des Realteiles des Ergebnisses


	a->real = omega;

	// kein Fehler aufgetreten

	return 0;
}

// Der ausfhrbare Quelltext der Funktion herf_M_S10

LRESULT herf_M_S10Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b,
						   LPCOMPLEXSCALAR c)

{	double herf;
	double breite;
	double moment;
	
	// bernahme der realen Teile der Argumente

	moment = b->real;
	breite = c->real;

	// Berechnung der erforderlichen Hhe

	herf = sqrt( (600*moment)/breite );

	// Rckgabe des Realteiles des Ergebnisses

	a->real = herf;

	// kein Fehler aufgetreten

	return 0;

}

// Der ausfhrbare Quelltext der Funktion herf_M_S13

LRESULT herf_M_S13Function(LPCOMPLEXSCALAR a, LPCOMPLEXSCALAR b,
						   LPCOMPLEXSCALAR c)

{	double herf;
	double breite;
	double moment;
	
	// bernahme der realen Teile der Argumente

	moment = b->real;
	breite = c->real;

	// Berechnung der erforderlichen Hhe

	herf = sqrt( (600*moment)/(1.3*breite) );

	// Rckgabe des Realteiles des Ergebnisses

	a->real = herf;

	// kein Fehler aufgetreten

	return 0;

}

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Alle MathCad-DLLs mssen einen DLL-Eintrittspunkt besitzen
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

extern "C" BOOL WINAPI _CRT_INIT(HINSTANCE hinstDLL, DWORD dwReason,
								 LPVOID lpReserved);

BOOL WINAPI  DllEntryPoint(HINSTANCE hDLL, DWORD dwReason, LPVOID lpReserved)
{
	switch (dwReason)
	{
		case DLL_PROCESS_ATTACH:

			if (!_CRT_INIT(hDLL, dwReason, lpReserved)) 
				return FALSE;

			if (CreateUserErrorMessageTable(hDLL, 4, 
				myErrorMessageTable))

			{	// Durchlauftrger ber 2 Felder, Gleichlast
				CreateUserFunction( hDLL, &DLT2_Amax );
				CreateUserFunction( hDLL, &DLT2_Amin );
				CreateUserFunction( hDLL, &DLT2_Mmax );
				CreateUserFunction( hDLL, &DLT2_Qmax );
				CreateUserFunction( hDLL, &DLT2_Mmin );

				// Durchlauftrger ber 3 Felder, Gleichlast
				CreateUserFunction( hDLL, &DLT3_Amax );
				CreateUserFunction( hDLL, &DLT3_Amin );
				CreateUserFunction( hDLL, &DLT3_Mmax );
				CreateUserFunction( hDLL, &DLT3_Qmax );
				CreateUserFunction( hDLL, &DLT3_Mmin );

				// Bemessung fr Biegung, Stahlbetonbau
				CreateUserFunction( hDLL, &As_M_B25_S500 );

				// Schubbemessung, Stahlbetonbau
				CreateUserFunction( hDLL, &As_Q_B25_S500 );

				// Biegebemessung Elastisch-Elastisch, St37, Stahlbetonbau
				CreateUserFunction( hDLL, &BemMy_IPE_St37 );
				CreateUserFunction( hDLL, &BemMy_HEB_St37 );

				// I-Profile suchen
				CreateUserFunction( hDLL, &Su_Iy_I );
			
				// Knickzahlen fr Nadelholz
				CreateUserFunction( hDLL, &omega );

				// Biegebemessung Nadelholz
				CreateUserFunction( hDLL, &herf_M_S10 );
				CreateUserFunction( hDLL, &herf_M_S13 );
			}

			break;

		case DLL_THREAD_ATTACH:
		case DLL_THREAD_DETACH:
		case DLL_PROCESS_DETACH:

			if (!_CRT_INIT(hDLL, dwReason, lpReserved)) 
				return FALSE;

			break;

	}

	return TRUE;

}

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+
//+ Ende 
//+
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
